-- ============================================================================
-- === AutoGateAktivatorSpecialization.lua
-- === Mod by [LSMT] Modding Team 
-- === LS25 /FS25
-- === Script by [LSMT] BaTt3RiE @ 2025
-- === Ver 1.0.0.3
-- ============================================================================

AutoGateAktivator = {}

local DEBUG = false
local modDirectory = g_currentModDirectory

local function dprint(...) if DEBUG then print("[AutoGateAktivator]", ...) end end

-- ============================================================================
-- KONFIGURATION - Hier kannst du alle Werte anpassen
-- ============================================================================
AutoGateAktivator.CONFIG = {
    -- Position und Größe
    offsetX = 0,      -- Seitliche Position (negativ = links, positiv = rechts)
    offsetY = 2,      -- Höhe über Boden
    offsetZ = 0,      -- Abstand vor dem Fahrzeug (negativ = hinter Fahrzeug)
    
    scaleX = 2,       -- Breite der Trigger-Box
    scaleY = 1,       -- Höhe der Trigger-Box
    scaleZ = 20,      -- FESTER Aktivierungsradius (in Metern)
    
    -- Collision-Werte wenn MOTOR AN und (Spieler ODER AutoDrive)
    collisionGroupActive = 0x100000,
    collisionMaskActive = 0x20000000,
    
    -- Collision-Werte wenn MOTOR AUS oder Standard-Helfer AN
    collisionGroupInactive = 0x0,
    collisionMaskInactive = 0x0,
    
    -- Sichtbarkeit (für Tests auf true setzen)
    visible = false,   -- true = rote Box sichtbar, false = unsichtbar
}

function AutoGateAktivator.prerequisitesPresent(_)
    return true
end

function AutoGateAktivator.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadTrigger", AutoGateAktivator.loadTrigger)
    SpecializationUtil.registerFunction(vehicleType, "unloadTrigger", AutoGateAktivator.unloadTrigger)
    SpecializationUtil.registerFunction(vehicleType, "updateCollisionValues", AutoGateAktivator.updateCollisionValues)
    SpecializationUtil.registerFunction(vehicleType, "shouldTriggerBeActive", AutoGateAktivator.shouldTriggerBeActive)
end

function AutoGateAktivator.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", AutoGateAktivator)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", AutoGateAktivator)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", AutoGateAktivator)
end

-- NEU: Prüft ob Trigger aktiv sein soll
-- AKTIV wenn: Motor AN UND (AutoDrive AN ODER Standard-Helfer AUS)
function AutoGateAktivator:shouldTriggerBeActive()
    -- Prüfe Motor-Status
    local isMotorStarted = self:getIsMotorStarted()
    
    -- Prüfe AutoDrive-Status ZUERST (hat Vorrang!)
    local isAutoDriveActive = false
    if self.ad ~= nil then
        -- Methode 1: stateModule:isActive()
        if self.ad.stateModule ~= nil and self.ad.stateModule.isActive ~= nil then
            isAutoDriveActive = self.ad.stateModule:isActive()
            dprint(string.format("AutoDrive stateModule:isActive(): %s", tostring(isAutoDriveActive)))
        end
        
        -- Methode 2: drivePathModule prüfen (zusätzliche Sicherheit)
        if self.ad.drivePathModule ~= nil and self.ad.drivePathModule.isTargetReached ~= nil then
            local isNotAtTarget = not self.ad.drivePathModule:isTargetReached()
            dprint(string.format("AutoDrive drivePathModule - nicht am Ziel: %s", tostring(isNotAtTarget)))
            -- AutoDrive ist aktiv wenn stateModule aktiv UND noch nicht am Ziel
            isAutoDriveActive = isAutoDriveActive and isNotAtTarget
        end
    end
    
    -- Prüfe Standard-Helfer-Status
    local isHelperActive = false
    if self.getIsAIActive ~= nil then
        isHelperActive = self:getIsAIActive()
        dprint(string.format("Standard-Helfer aktiv: %s", tostring(isHelperActive)))
    end
    
    -- Bedingung 1: Motor muss laufen
    if not isMotorStarted then
        dprint("Motor AUS -> Trigger INAKTIV")
        return false
    end
    
    -- Bedingung 2: AutoDrive hat VORRANG - wenn aktiv, dann immer Trigger AKTIV
    if isAutoDriveActive then
        dprint("Motor AN, AutoDrive AKTIV -> Trigger AKTIV (ignoriert Standard-Helfer)")
        return true
    end
    
    -- Bedingung 3: Kein AutoDrive -> Standard-Helfer darf NICHT aktiv sein
    if isHelperActive then
        dprint("Motor AN, AutoDrive AUS, Standard-Helfer AKTIV -> Trigger INAKTIV")
        return false
    end
    
    -- Motor AN, AutoDrive AUS, Standard-Helfer AUS -> Spieler fährt
    dprint("Motor AN, AutoDrive AUS, Standard-Helfer AUS, Spieler fährt -> Trigger AKTIV")
    return true
end

-- NEU: Aktualisiert die Collision-Werte basierend auf Motor-Status UND Helfer-Status
function AutoGateAktivator:updateCollisionValues(shouldBeActive)
    local spec = self.spec_autoGateAktivator
    if spec == nil or spec.attachedI3d == nil then
        return
    end
    
    local cfg = AutoGateAktivator.CONFIG
    local collisionGroup, collisionMask
    
    if shouldBeActive then
        collisionGroup = cfg.collisionGroupActive
        collisionMask = cfg.collisionMaskActive
        dprint("Trigger AKTIV (Motor AN, Standard-Helfer AUS)")
    else
        collisionGroup = cfg.collisionGroupInactive
        collisionMask = cfg.collisionMaskInactive
        dprint("Trigger INAKTIV (Motor AUS oder Standard-Helfer AN)")
    end
    
    -- Rekursiv durch i3D gehen und Collision-Werte setzen
    local function updateNodeCollision(node)
        if node == nil or node == 0 then return end
        
        if getHasClassId(node, ClassIds.SHAPE) then
            setCollisionFilterGroup(node, collisionGroup)
            setCollisionFilterMask(node, collisionMask)
            dprint(string.format("  Shape aktualisiert: Group=0x%x, Mask=0x%x", collisionGroup, collisionMask))
            return
        end
        
        local numChildren = getNumOfChildren(node)
        for i = 0, numChildren - 1 do
            updateNodeCollision(getChildAt(node, i))
        end
    end
    
    updateNodeCollision(spec.attachedI3d)
end

-- Rekursive Funktion zum Finden und Konfigurieren von Shapes
local function configureShapeRecursive(node, depth, collisionGroup, collisionMask)
    if node == nil or node == 0 then return false end
    
    depth = depth or 0
    if depth > 10 then return false end
    
    if getHasClassId(node, ClassIds.SHAPE) then
        dprint(string.format("  Shape gefunden: %s (Ebene %d)", getName(node), depth))
        
        local rigidBodyType = getRigidBodyType(node)
        dprint(string.format("    Aktueller Rigid Body Type: %s", tostring(rigidBodyType)))
        
        if rigidBodyType == 0 then
            dprint("    Füge Dynamic Rigid Body Component hinzu...")
            addRigidBodyComponent(node, 3)
        elseif rigidBodyType ~= 3 then
            dprint(string.format("    Setze Rigid Body Type von %d auf 3 (Dynamic)", rigidBodyType))
            setRigidBodyType(node, 3)
        else
            dprint("    Rigid Body ist bereits Dynamic")
        end
        
        setCollisionFilterGroup(node, collisionGroup)
        setCollisionFilterMask(node, collisionMask)
        
        dprint(string.format("    Collision-Werte gesetzt:"))
        dprint(string.format("      Group: %d (0x%x)", collisionGroup, collisionGroup))
        dprint(string.format("      Mask:  %d (0x%x)", collisionMask, collisionMask))
        
        return true
    end
    
    local numChildren = getNumOfChildren(node)
    for i = 0, numChildren - 1 do
        local child = getChildAt(node, i)
        if configureShapeRecursive(child, depth + 1, collisionGroup, collisionMask) then
            return true
        end
    end
    
    return false
end

function AutoGateAktivator:loadTrigger()
    local spec = self.spec_autoGateAktivator
    if spec == nil then
        return false
    end
    
    local cfg = AutoGateAktivator.CONFIG
    
    local i3dPath = "i3d/aktivator.i3d"
    local i3dFile = Utils.getFilename(i3dPath, modDirectory)

    dprint("Lade i3D:", i3dFile)
    local root = loadI3DFile(i3dFile, false, false, false)
    if root == 0 then
        print("[AutoGateAktivator] FEHLER: i3D konnte nicht geladen werden:", i3dFile)
        return false
    end

    link(self.rootNode, root)
    
    setTranslation(root, cfg.offsetX, cfg.offsetY, cfg.offsetZ)
    setScale(root, cfg.scaleX, cfg.scaleY, cfg.scaleZ)
    setRotation(root, 0, 0, 0)
    
    -- Setze Sichtbarkeit rekursiv
    local function setVisibilityRecursive(node, visible)
        if node == nil or node == 0 then return end
        setVisibility(node, visible)
        
        local numChildren = getNumOfChildren(node)
        for i = 0, numChildren - 1 do
            local child = getChildAt(node, i)
            setVisibilityRecursive(child, visible)
        end
    end
    
    setVisibilityRecursive(root, cfg.visible)
    spec.attachedI3d = root

    -- Initiale Collision basierend auf Motor UND Helfer-Status
    local shouldBeActive = self:shouldTriggerBeActive()
    local collisionGroup = shouldBeActive and cfg.collisionGroupActive or cfg.collisionGroupInactive
    local collisionMask = shouldBeActive and cfg.collisionMaskActive or cfg.collisionMaskInactive

    dprint("=== KONFIGURATION ===")
    dprint(string.format("Position: X=%.1f, Y=%.1f, Z=%.1f", cfg.offsetX, cfg.offsetY, cfg.offsetZ))
    dprint(string.format("Skalierung: Breite=%.1f, Höhe=%.1f, Länge=%.1f", cfg.scaleX, cfg.scaleY, cfg.scaleZ))
    dprint(string.format("Trigger aktiv: %s", tostring(shouldBeActive)))
    dprint(string.format("Collision Group: %d (0x%x)", collisionGroup, collisionGroup))
    dprint(string.format("Collision Mask: %d (0x%x)", collisionMask, collisionMask))

    local shapeFound = configureShapeRecursive(root, 0, collisionGroup, collisionMask)
    
    if not shapeFound then
        print("[AutoGateAktivator] WARNUNG: Keine Shape in der i3D gefunden!")
    end

    g_currentMission:addNodeObject(root, self)

    dprint("i3D angehängt, Collision konfiguriert und registriert")
    print("[AutoGateAktivator] === BEREIT === Trigger öffnet Tore für Spieler UND AutoDrive (nicht für Standard-Helfer)")
    
    return true
end

function AutoGateAktivator:unloadTrigger()
    local spec = self.spec_autoGateAktivator
    if spec == nil then
        return
    end
    
    if spec.attachedI3d ~= nil then
        g_currentMission:removeNodeObject(spec.attachedI3d)
        delete(spec.attachedI3d)
        spec.attachedI3d = nil
        dprint("Trigger entladen")
    end
end

function AutoGateAktivator:onLoad(_)
    self.spec_autoGateAktivator = self.spec_autoGateAktivator or {}
    local spec = self.spec_autoGateAktivator
    
    -- Initialer Status
    spec.lastTriggerState = nil
    
    -- Lade Trigger (wird auf allen Clients UND Server geladen)
    self:loadTrigger()
end

function AutoGateAktivator:onUpdate(dt)
    local spec = self.spec_autoGateAktivator
    if spec == nil then
        return
    end
    
    -- Prüfe ob Trigger aktiv sein soll
    local shouldBeActive = self:shouldTriggerBeActive()
    
    -- Wenn Status sich geändert hat
    if spec.lastTriggerState ~= shouldBeActive then
        dprint(string.format("[%s] Trigger-Status geändert: %s -> %s", 
            g_server and "SERVER" or "CLIENT",
            tostring(spec.lastTriggerState), 
            tostring(shouldBeActive)))
        
        self:updateCollisionValues(shouldBeActive)
        spec.lastTriggerState = shouldBeActive
    end
end

function AutoGateAktivator:onDelete()
    self:unloadTrigger()
end