-- ============================================================================
-- === AutoGate.Lua
-- === Mod by [LSMT] Modding Team 
-- === LS25 /FS25
-- === Script by [LSMT] BaTt3RiE @ 2025
-- === Ver 1.0.1.0
-- ============================================================================

local DEBUG = false -- Setze auf true, um Debug-Logs zu aktivieren

OPEN_ON_ENTER   = true     -- sofort öffnen
CLOSE_ON_LEAVE  = true     -- beim Verlassen schließen
CHECK_OWNERSHIP = true     -- true = nur Grundstücksbesitzer können Tor nutzen
CLOSE_DELAY_MS  = 0        -- 0  = erst nach x ms schließen

local closeTimers = {}
local lastShownTimes = {}
local objectsInTrigger = {}
local doorClosingInitiated = {}
local doorOpenInitiated = {}

local function debugPrint(...)
    if DEBUG then
        print(...)
    end
end

local function setDoorDirection(animatedObject, dir)
    if animatedObject.animation.direction ~= dir then
        animatedObject:setDirection(dir)
    end
end

local function findGameObject(nodeId)
    while nodeId ~= 0 do
        local obj = g_currentMission.nodeToObject[nodeId]
        if obj ~= nil then
            return obj
        end
        nodeId = getParent(nodeId)
    end
    return nil
end

local function autoDoor_triggerCallback(self, superFunc, triggerId, otherId,
                                        onEnter, onLeave, onStay, anotherID)

    -- Original (Activatable-Handling etc.)
    if superFunc ~= nil then
        superFunc(self, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
    end

    if not self.isEnabled then return end

    -- Spieler?
    local isPlayer = (g_currentMission.players[otherId] ~= nil)
    local playerFarmId = nil
    if isPlayer then
        playerFarmId = g_currentMission.players[otherId].farmId
    end

    -- Fahrzeug / Gerät / Anhänger?
    local vehicleObj = findGameObject(otherId)       -- rekursive Suche
    local isVehicle  = (vehicleObj ~= nil) and vehicleObj:isa(Vehicle)
    local vehicleFarmId = nil
    if isVehicle then
        vehicleFarmId = vehicleObj:getOwnerFarmId()
    end

    if not (isPlayer or isVehicle) then return end

    local hasAccess = true -- Standard: Zugriff erlaubt, falls CHECK_OWNERSHIP deaktiviert ist
    
    -- Nur Besitzprüfung durchführen, wenn aktiviert
    if CHECK_OWNERSHIP then
        -- Bestimme den Besitzer des Landes, auf dem das Tor steht
        local x, _, z = getWorldTranslation(self.nodeId)
        local landOwnerId = g_farmlandManager:getFarmlandOwner(g_farmlandManager:getFarmlandIdAtWorldPosition(x, z))

        -- Bestimme die Farm-ID, die das Objekt besitzt
        local objectFarmId = nil
        if isPlayer then
            objectFarmId = playerFarmId
        elseif isVehicle then
            objectFarmId = vehicleFarmId
        end

        -- Prüfe, ob das Objekt Zugriff auf dieses Land hat
        hasAccess = false -- Standardmäßig kein Zugriff, wenn Prüfung aktiv
        if landOwnerId == FarmlandManager.NO_OWNER_FARM_ID then
            hasAccess = false                       -- Kein Besitzer, also kein Zugriff         
        elseif objectFarmId ~= nil then
            if objectFarmId == landOwnerId then
                hasAccess = true                    -- Grundstücksbesitzer hat Zugriff  
            elseif g_currentMission.accessHandler:canFarmAccessOtherId(objectFarmId, landOwnerId) then
                hasAccess = true                    -- Farm-Mitglied hat Zugriff
            end
        end
    end

    -- Ohne Berechtigung nicht öffnen
    if not hasAccess then
        print("Zugriff verweigert: Tor öffnet sich nur für Grundstücksbesitzer oder Hofmitglieder!")
        return
    end

    -- Initialisiere Zähler für dieses Tor, falls noch nicht vorhanden
    if objectsInTrigger[self] == nil then
        objectsInTrigger[self] = 0
    end

    if onEnter and OPEN_ON_ENTER then
        closeTimers[self] = nil                      -- laufenden Timer stoppen
        doorClosingInitiated[self] = nil            -- Schließstatus zurücksetzen
        objectsInTrigger[self] = objectsInTrigger[self] + 1
        debugPrint("Objekt betritt Tor - Objekte im Trigger: " .. objectsInTrigger[self])

        if not doorOpenInitiated[self] then
            setDoorDirection(self, 1)
            doorOpenInitiated[self] = true  -- <- MERKE DIR, DASS DAS ÖFFNEN EINGELEITET WURDE
        end

        return
    end

    if onLeave and CLOSE_ON_LEAVE then
        objectsInTrigger[self] = math.max(0, objectsInTrigger[self] - 1)
        debugPrint("Objekt verlässt Tor - Objekte im Trigger: " .. objectsInTrigger[self])
        
        -- Nur schließen, wenn kein Objekt mehr im Trigger ist
        if objectsInTrigger[self] == 0 then
            if CLOSE_DELAY_MS == 0 then
                setDoorDirection(self, -1)
                doorOpenInitiated[self] = false  -- <- ZURÜCKSETZEN FÜR NÄCHSTES MAL
                debugPrint("Tor wird sofort geschlossen")
            else
                -- Verwende den bereits implementierten Timer-Mechanismus statt addDelayedCall
                closeTimers[self] = g_currentMission.time + CLOSE_DELAY_MS
                debugPrint("Timer gestartet: Tor schließt in " .. (CLOSE_DELAY_MS / 1000) .. " Sekunden")
                -- Halte das Objekt aktiv, damit update aufgerufen wird
                self:raiseActive()
            end
        end
    end
end

AnimatedObject.triggerCallback = Utils.overwrittenFunction(
    AnimatedObject.triggerCallback,
    autoDoor_triggerCallback
)

local function autoDoor_update(self, superFunc, dt)
    if superFunc ~= nil then
        superFunc(self, dt)
    end

    -- Nur Server bearbeitet Timer, Clients kriegen State via Sync
    if g_server == nil then return end

    local targetTime = closeTimers[self]
    if targetTime ~= nil then
        -- Tor aktiv halten, bis es geschlossen ist
        self:raiseActive()
        
        if g_currentMission.time >= targetTime then
            -- NUR EINMAL den Schließvorgang einleiten
            if self.animation ~= nil and self.animation.direction ~= -1 and not doorClosingInitiated[self] then
                debugPrint("Tor-Timer abgelaufen - Starte Schließvorgang")
                setDoorDirection(self, -1)
                doorClosingInitiated[self] = true  -- <- MERKE DIR, DASS DAS SCHLIEẞEN EINGELEITET WURDE
                doorOpenInitiated[self] = false
            end
            
            -- Prüfung, ob das Tor vollständig geschlossen ist
            if self.animation ~= nil and self.animation.animTime ~= nil and self.animation.animTime >= 0.99 then
                closeTimers[self] = nil
                lastShownTimes[self] = nil
                doorClosingInitiated[self] = nil  -- <- ZURÜCKSETZEN FÜR NÄCHSTES MAL
                doorOpenInitiated[self] = nil     -- <- ZURÜCKSETZEN FÜR NÄCHSTES MAL
                debugPrint("Tor ist vollständig geschlossen")
            end
        else
            -- Timer-Countdown anzeigen...
            local timeLeft = math.floor((targetTime - g_currentMission.time) / 1000)
            
            if lastShownTimes[self] ~= timeLeft then
                lastShownTimes[self] = timeLeft
                debugPrint("Tor schließt in " .. timeLeft .. " Sekunden")
            end
        end
    end
end

AnimatedObject.update = Utils.overwrittenFunction(
    AnimatedObject.update,
    autoDoor_update
)